/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Workfile: $
   Last Modification:
    $Author: sspies $
    $Modtime: $
    $Revision: 673 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: no
     WinCE        : no
 
   Description:
    Definition file of the TCP Connector configuration functions
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     2         27.11.09    SS       Review
     1         27.08.09    SS       created 


**************************************************************************************/

/*****************************************************************************/
/*! \file TCPConfig.h
*   Definition file of the TCP Connector configuration functions             */
/*****************************************************************************/

/* prevent multiple inclusion */
#pragma once

#include "ConnectorConfig.h"

/*****************************************************************************/
/*! \addtogroup netX_CONNECTOR_TCP netX TCP Connector                        */
/*! \{                                                                       */
/*****************************************************************************/

extern const UUID g_tConnectorUUID;

///////////////////////////////////////////////////////////////////////////////////////////
/// TCP configuration keys
///////////////////////////////////////////////////////////////////////////////////////////
#define NXCON_TCPKEY_IPBEGIN        "IPADDR"         /*!< Configuration key to adjust the IP address (begin of range) of the TCP interface */
#define NXCON_TCPKEY_IPEND          "IPADDREND"      /*!< Configuration key to adjust the IP address (end of range) of the TCP interface   */
#define NXCON_TCPKEY_PORT           "PORT"           /*!< Configuration key to adjust the port of the TCP interface                        */
#define NXCON_TCPKEY_CONNECTTIMEOUT "CONNECTTIMEOUT" /*!< Configuration key to adjust the connection timeout of the TCP interface          */

///////////////////////////////////////////////////////////////////////////////////////////
/// TCP configuration class
///////////////////////////////////////////////////////////////////////////////////////////
class CTCPConfig:public CConnectorConfig
{
public:
  // Constructor
  CTCPConfig( BOOL fLoadFormRegistry = FALSE);

  // Copy Constructor
  CTCPConfig( const CTCPConfig& cTCPConfig) {*this = cTCPConfig;}
  
  // Destructor
	~CTCPConfig();
  
  // Assignment operator
  CTCPConfig& operator= ( const CTCPConfig& cCTCPConfig) {CConnectorConfig::operator=( cCTCPConfig); return *this;}

  DWORD  GetConnectTimeout ( void);

  ///////////////////////////////////////////////////////////////////////////////////////////
  /// Iterate over configured TCP interfaces 
  /// Note: A single interface configuration entry can compass a whole IP address range.
  ///////////////////////////////////////////////////////////////////////////////////////////
  class iterator:public CConnectorConfig::iterator
  {

  public:
    
    // Constructor
    iterator(INTFCONFIG_MAP::iterator iterIntf)
    :CConnectorConfig::iterator(iterIntf)
    {
    }
    
    // Copy Constructor
    iterator(const CConnectorConfig::iterator& iterConfig)
    :CConnectorConfig::iterator(iterConfig)
    {
    }      

    /////////////////////////////////////////////////////////////////////////////
    /// Returns TCP port 
    /// \return TCP Port (Network byte order)
    /////////////////////////////////////////////////////////////////////////////
    unsigned short GetTCPPort ( void)
    {
      unsigned short usRet = 0;

      /* Search for TCP port config key */
      CONFIGKEY_MAP::iterator iterKey = m_iterIntf->second.find( NXCON_TCPKEY_PORT);
      if (m_iterIntf->second.end() != iterKey)
      {
        usRet = htons((unsigned short)_ttoi(iterKey->second));
      }

      return usRet;
    }

    /////////////////////////////////////////////////////////////////////////////
    /// Get first IP address of the configured IP address range
    /// \return IP address (Host byte order)
    /////////////////////////////////////////////////////////////////////////////
    DWORD GetIPRangeBegin ( void)
    {
      DWORD dwIPBegin = INADDR_NONE;

      /* Search for IP range begin config key */
      CONFIGKEY_MAP::iterator iterKey = m_iterIntf->second.find( NXCON_TCPKEY_IPBEGIN);
      if (m_iterIntf->second.end() != iterKey)
      {
        dwIPBegin = ntohl(inet_addr(iterKey->second));
      }

      return dwIPBegin;
    }

    /////////////////////////////////////////////////////////////////////////////
    /// Get last IP address of the configured IP address range
    /// \return IP address (Host byte order)
    /////////////////////////////////////////////////////////////////////////////
    DWORD GetIPRangeEnd ( void)
    {
      DWORD dwIPBegin  = INADDR_NONE;
      DWORD dwIPEnd    = INADDR_NONE;

      dwIPBegin = GetIPRangeBegin();
      
      /* Search for IP range end config key */
      CONFIGKEY_MAP::iterator iterKey = m_iterIntf->second.find( NXCON_TCPKEY_IPEND);
      if (m_iterIntf->second.end() != iterKey)
      {
        dwIPEnd = ntohl(inet_addr(iterKey->second));
        if (dwIPEnd<dwIPBegin)
          dwIPEnd = dwIPBegin;

      } else
      {
        dwIPEnd = dwIPBegin;
      }

      return dwIPEnd;
    }

    /////////////////////////////////////////////////////////////////////////////
    /// Get number of IP addresses in configured IP address range
    /// \return Number of IP addresses
    /////////////////////////////////////////////////////////////////////////////
    DWORD GetIPRangeSize ( void)
    {
      return (GetIPRangeEnd()-GetIPRangeBegin())+1;
    }
  };

  /////////////////////////////////////////////////////////////////////////////
  /// Insert TCP interface in configuration
  /// \param dwIPRangeBegin Begin of range IP (Network byte order)
  /// \param dwIPRangeEnd   End of range IP (Network byte order)
  /// \param usTCPPort      TCP port (Network byte order)
  /// \return iterator
  /////////////////////////////////////////////////////////////////////////////
  iterator insert( DWORD dwIPRangeBegin, DWORD dwIPRangeEnd, USHORT usTCPPort)
  {
    if (dwIPRangeBegin == dwIPRangeEnd)
    {
      dwIPRangeEnd = 0;
    }
    in_addr tIPAddrBegin;
    in_addr tIPAddrEnd;
    tIPAddrBegin.S_un.S_addr = dwIPRangeBegin;
    tIPAddrEnd.S_un.S_addr = dwIPRangeEnd;

    CString szIPRangeBegin;
    CString szIPRangeEnd;
    CString szTCPPort;
    szIPRangeBegin.Format("%s", inet_ntoa(tIPAddrBegin));
    szIPRangeEnd.Format("%s", inet_ntoa(tIPAddrEnd));
    szTCPPort.Format("%d", ntohs(usTCPPort));

    CONFIGKEY_MAP cmKeyMap;
    cmKeyMap.insert(make_pair(NXCON_TCPKEY_PORT, szTCPPort));
    cmKeyMap.insert(make_pair(NXCON_TCPKEY_IPBEGIN, szIPRangeBegin));
    cmKeyMap.insert(make_pair(NXCON_TCPKEY_IPEND, szIPRangeEnd));

    iterator iterNew    = m_cmIntfConfigMap.end();
    CString  csIntfName = CreateInterfaceName();
    if ( NXCON_NO_ERROR == CreateInterfaceConfig( csIntfName))
    {
      StoreInterfaceConfig(csIntfName, cmKeyMap);
      iterNew = m_cmIntfConfigMap.find( csIntfName);
    }

    return iterNew;
  }

  /////////////////////////////////////////////////////////////////////////////
  /// Find element in configuration via interface name
  /// \param  szName Name of interface to find
  /// \return Iterator
  /////////////////////////////////////////////////////////////////////////////
  iterator find( const char* szName)
  {
    return iterator(m_cmIntfConfigMap.find( szName));
  }   

  /////////////////////////////////////////////////////////////////////////////
  /// Find element in configuration via IP address and TCP port
  /// \param  dwIPAddress IP address of interface to find (Network byte order)
  /// \param  usTCPPort   TCP port of interface to find (Network byte order)
  /// \return Iterator
  /////////////////////////////////////////////////////////////////////////////
  iterator find( DWORD dwIPAddress, USHORT usTCPPort)
  {
    iterator iterRet = m_cmIntfConfigMap.end();

    /* Enumerate configured interfaces and create interface objects */
    for (iterator iterConfig = m_cmIntfConfigMap.begin(); iterConfig != m_cmIntfConfigMap.end(); ++iterConfig)
    {
      if (!iterConfig.IsExcluded())
      {
        /* Enumerate configured interfaces and create interface objects */
        for (DWORD dwIPCurrent =  iterConfig.GetIPRangeBegin(); dwIPCurrent <=  iterConfig.GetIPRangeEnd(); ++dwIPCurrent)
        {
          if ( (htonl(dwIPCurrent)      == dwIPAddress) &&
               (iterConfig.GetTCPPort() == usTCPPort  )    )
          {
            iterRet = iterConfig;
            break;
          }
        }
            
        if (iterRet != m_cmIntfConfigMap.end())
          break;
      }
    }

    return iterRet;
  }

protected:

};

/*****************************************************************************/
/*! \}                                                                       */
/*****************************************************************************/


